package.add("src")
require("instruments")

local sim = ac.getSim()

local stored = ac.storage({
	activeDisplay = 1,
	splashShown = false,
	initialized = false,
})

stored.splashShown = false
stored.initialized = false

local last = {}

local displaySize = vec2(920, 920 / 2.15)

local xOffset = 3168
local yOffset = 3897

local dashImage = ui.decodeImage(io.loadFromZip(ac.findFile("assets.zip"), "8_seg_background.png"))
local dashBacklitImage = ui.decodeImage(io.loadFromZip(ac.findFile("assets.zip"), "8_seg_background_backlit.png"))

local rpmBarImages = {}

for i = 0, 60 do
	rpmBarImages[i] = ui.decodeImage(io.loadFromZip(ac.findFile("assets.zip"), "rpm_bar_" .. i .. ".png"))
end

local COMMON_INSTRUMENTS = function()
	local lastRpm = car.rpm
	local lastRpmBar = 0
	local timerRpmBar = 0

	function drawRpmGauge(dt, fdata)
		local rpmMult = 100
		local rpmAdd = 0

		for i = 0, 60 do
			if i < 10 then
				rpmMult = 300
			else
				rpmMult = 100
				rpmAdd = 2100
			end

			if fdata.rpm > i * rpmMult + rpmAdd then
				if fdata.rpm > lastRpm then
					lastRpmBar = i
				end
				display.image({
					image = rpmBarImages[i],
					pos = vec2(2685, 3662),
					size = displaySize,
					color = rgbm.colors.black,
					uvStart = vec2(0, 0),
					uvEnd = vec2(1, 1),
				})
			end
		end

		if fdata.rpm < lastRpm then
			timerRpmBar = os.clock() + 1
		end

		if timerRpmBar > os.clock() then
			display.image({
				image = rpmBarImages[lastRpmBar],
				pos = vec2(2685, 3662),
				size = displaySize,
				color = rgbm.colors.black,
				uvStart = vec2(0, 0),
				uvEnd = vec2(1, 1),
			})
		end

		lastRpm = fdata.rpm
	end

	return function(dt, fdata)
		if car.headlightsActive then
			display.image({ image = dashBacklitImage, pos = vec2(2685, 3662), size = displaySize })
		else
			display.image({ image = dashImage, pos = vec2(2685, 3662), size = displaySize })
		end

		drawGear(dt, fdata)
		drawRpmGauge(dt, fdata)
		drawRpmLabels()
	end
end

local commonInstruments = COMMON_INSTRUMENTS()

local DASH_ANALYSIS = function()
	return function(dt, fdata)
		commonInstruments(dt, fdata)

		display.text({
			text = "ANALYSIS",
			pos = vec2(xOffset + 300, yOffset + 70),
			letter = vec2(10, 25),
			font = "vrc-arial-bold",
			width = 100,
			alignment = 0.5,
			color = 0,
		})

		drawLapTime(dt, fdata)
		drawRpm(dt, fdata)
		drawLap(dt, fdata)
	end
end

local DASH_RACE_1 = function()
	return function(dt, fdata)
		commonInstruments(dt, fdata)
		display.text({
			text = "RACE",
			pos = vec2(xOffset + 300, yOffset + 6),
			letter = vec2(11.5, 25),
			font = "vrc-arial-bold",
			width = 100,
			alignment = 0.5,
			color = 0,
		})

		drawSpeed(dt, fdata)
		drawFuel(dt, fdata)
		drawOilPressure(dt, fdata)
		-- drawBottomInfo(" DELTA  ")
		-- drawDelta(dt, fdata)
	end
end

local DASH_RACE_2 = function()
	return function(dt, fdata)
		commonInstruments(dt, fdata)
		display.text({
			text = "RACE",
			pos = vec2(xOffset + 300, yOffset + 6),
			letter = vec2(11.5, 25),
			font = "vrc-arial-bold",
			width = 100,
			alignment = 0.5,
			color = 0,
		})

		drawLapTime(dt, fdata)
		drawFuel(dt, fdata)
		drawLap(dt, fdata)
	end
end

local DASH_PRACTICE = function()
	return function(dt, fdata)
		commonInstruments(dt, fdata)
		display.text({
			text = "PRACTICE",
			pos = vec2(xOffset + 300, yOffset + 27),
			letter = vec2(10.9, 25),
			font = "vrc-arial-bold",
			width = 100,
			alignment = 0.5,
			color = 0,
		})

		drawRpm(dt, fdata)
		drawBrakeBias(dt, fdata)
		drawSpeed(dt, fdata)
	end
end

local DASH_WARMUP = function()
	return function(dt, fdata)
		commonInstruments(dt, fdata)
		display.text({
			text = "WARM",
			pos = vec2(xOffset + 280, yOffset + 50),
			letter = vec2(12, 25),
			font = "vrc-arial-bold",
			width = 100,
			alignment = 0.5,
			color = 0,
		})
		display.text({
			text = "UP",
			pos = vec2(xOffset + 336, yOffset + 50),
			letter = vec2(12, 25),
			font = "vrc-arial-bold",
			width = 100,
			alignment = 0.5,
			color = 0,
		})

		drawRpm(dt, fdata)
		drawOilPressure(dt, fdata)
		drawFuelPressure(dt, fdata)
	end
end

local OFFLINE = function()
	return function(dt, fdata)
		display.image({ image = dashImage, pos = vec2(2685, 3662), size = displaySize })
	end
end

local SPLASH = function()
	return function(dt, fdata)
		drawBottomInfo("     VRC")
	end
end

local PIT_LIMITER = function()
	return function(dt, fdata)
		drawBottomInfo(" PIT LIMITER")
	end
end

local LOW_FUEL = function()
	return function(dt, fdata)
		drawBottomInfo("LOW FUEL " .. string.pad(math.round(fdata.fuel), 4, " ", -1))
	end
end

local LOW_OIL_PRESSURE = function()
	return function(dt, fdata)
		drawBottomInfo("LOW OIL PRESS")
	end
end

local BRAKE_BIAS = function()
	return function(dt, fdata)
		drawBottomInfo(string.format("    BBIAS %.1f", fdata.brakeBias))
	end
end

local LAST_LAP_VALID = function()
	return function(dt, fdata)
		drawBottomInfo("INVALID LAP")
	end
end

local NEW_BEST_LAP = function()
	return function(dt, fdata)
		drawBottomInfo(" NEW BEST LAP")
	end
end

local LAST_LAP_TIME = function()
	return function(dt, fdata)
		drawBottomInfo("LASTLAP")
		drawLastLapTimeBottom(dt, fdata)
	end
end

local WT_OT = function()
	return function(dt, fdata)
		drawBottomInfo(" WT " .. math.round(fdata.waterTemperature))
		drawBottomInfo("       OT " .. math.round(fdata.oilTemperature))
	end
end

local DELTA = function()
	return function(dt, fdata)
		drawBottomInfo("DELTA")
		drawDelta(dt, fdata)
	end
end

local pages = {
	DASH_RACE_1(),
	DASH_RACE_2(),
	DASH_PRACTICE(),
	DASH_WARMUP(),
	DASH_ANALYSIS(),
}

local popups = {
	OFFLINE(),
	SPLASH(),
	PIT_LIMITER(),
	LOW_FUEL(),
	LOW_OIL_PRESSURE(),
	BRAKE_BIAS(),
	LAST_LAP_VALID(),
	NEW_BEST_LAP(),
	LAST_LAP_TIME(),
	WT_OT(),
	DELTA(),
}

local popupVariables = {
	OFFLINE = { 1, 0 },
	SPLASH = { 2, 1 },
	PIT_LIMITER = { 3, 3 },
	LOW_FUEL = { 4, 1 },
	LOW_OIL_PRESSURE = { 5, 1 },
	BRAKE_BIAS = { 6, 3 },
	LAST_LAP_VALID = { 7, 3 },
	NEW_BEST_LAP = { 8, 1 },
	LAST_LAP_TIME = { 9, 3 },
	WT_OT = { 10, 0 },
	DELTA = { 11, 0 },
}

local pageController = function()
	local currentPage = 1

	local pageDownButton = ac.ControlButton("__EXT_LIGHT_E")
	local pageUpButton = ac.ControlButton("__EXT_LIGHT_F")

	pageDownButton:onPressed(function()
		if currentPage == 1 then
			currentPage = #pages
		else
			currentPage = currentPage - 1
		end

		stored.activeDisplay = currentPage
	end)

	pageUpButton:onPressed(function()
		if currentPage >= #pages then
			currentPage = 1
		else
			currentPage = currentPage + 1
		end

		stored.activeDisplay = currentPage
	end)

	return function(forcedMode)
		if forcedMode then
			return forcedMode
		end

		if car.isAIControlled then
			if sim.raceSessionType == ac.SessionType.Race then
				currentPage = 1
			else
				currentPage = 3
			end

			stored.activeDisplay = currentPage
			return currentPage
		end

		if not stored.initialized then
			stored.initialized = true
		end

		return currentPage
	end
end
local popupController = function()
	local timer = 0
	local popupQueue = {}
	local currentPopup = nil

	local function triggerPopup(item, value, screen, popupTime, condition)
		if last[item] == nil then
			last[item] = value
		end

		if value ~= nil and last[item] ~= value or condition then
			last[item] = value
			return item
		end

		return nil
	end
	return function(fdata, page, forcedMode)
		if sim.isInMainMenu or (car.isAIControlled and car.splinePosition > 0.1) then
			last["LAST_LAP_TIME"] = car.lapCount
		end

		if last["NEW_BEST_LAP"] ~= car.bestLapTimeMs and not car.isLastLapValid then
			last["NEW_BEST_LAP"] = car.bestLapTimeMs
		end

		local popupTriggers = {
			triggerPopup("PIT_LIMITER", nil, 3, 3, fdata.isInPitlane or car.manualPitsSpeedLimiterEnabled),
			triggerPopup("LOW_FUEL", nil, 4, 1, fdata.fuel < 6),
			triggerPopup("LOW_OIL_PRESSURE", nil, 5, 1, fdata.oilPressure / 10 < 32 and currentPopup ~= 4),
			triggerPopup("BRAKE_BIAS", fdata.brakeBias, 6, 3),
			triggerPopup("LAST_LAP_VALID", car.isLastLapValid, 7, 3),
			triggerPopup("NEW_BEST_LAP", car.bestLapTimeMs, 8, 1),
			triggerPopup("LAST_LAP_TIME", car.lapCount, 9, 3),
		}

		for _i, popup in pairs(popupTriggers) do
			if popup and not table.contains(popupQueue, popup) then
				table.insert(popupQueue, popup)
			end
		end

		if timer < os.clock() then
			if #popupQueue > 0 then
				currentPopup = popupVariables[popupQueue[1]][1]
				timer = os.clock() + popupVariables[popupQueue[1]][2]
				table.remove(popupQueue, 1)
			end
		end

		if not fdata.isIgnitionOn then
			return 1
		elseif fdata.isIgnitionOn and not fdata.isElectronicsBooted then
			return 2
		elseif timer >= os.clock() then
			return currentPopup
		elseif page ~= 2 then
			return 10
		else
			return 11
		end
	end
end

local imagesLoaded = false
local function imagesInitialized()
	if not imagesLoaded then
		local loadedTest = true

		for i = 0, 60 do
			if not ui.isImageReady(rpmBarImages[i]) then
				loadedTest = false
			end
		end

		if not ui.isImageReady(dashImage) then
			loadedTest = false
		end

		if not ui.isImageReady(dashBacklitImage) then
			loadedTest = false
		end

		imagesLoaded = loadedTest

		return imagesLoaded
	else
		return true
	end
end

local getPage = pageController()
local getPopup = popupController()
function displayController(dt, fdata, forcedMode)
	if not imagesInitialized() then
		return
	end

	local page = getPage(forcedMode)

	pages[page](dt, fdata)

	local popup = getPopup(fdata, page, forcedMode)
	if popup then
		popups[popup](dt, fdata)
	end
end
